//
//  GSGlyph.h
//  FontBezier Doc
//
//  Created by Georg Seifert on 20.10.05.
//  Copyright 2005 schriftgestaltung.de. All rights reserved.
//

#import <Cocoa/Cocoa.h>
#import <GlyphsCore/GSUserDataProtocol.h>
@class GSFont;
@class GSLayer;
@class GSUndoManager;
@class MGOrderedDictionary;
@class GSInstance;
@class GSPartProperty;
@class GSGlyphInfo;

typedef NS_ENUM(NSInteger, GSGlyphCopyOptions) {
	GSGlyphCopyNoLayers = 0,
	GSGlyphCopySpecialLayers = 1 << 1,
	GSGlyphCopyAllLayers = 2 << 1,
};

/**
 The class defining the glyph object
*/

NS_ASSUME_NONNULL_BEGIN

@interface GSGlyph : NSObject <NSCoding, NSCopying, GSUserDataProtocol> {
	MGOrderedDictionary *_layers;
	NSString *_name;
	NSMutableOrderedSet *_unicodes;
	NSString *_script;
	NSString *_note;
	NSString *_category;
	NSString *_subCategory;
	NSString *_leftMetricsKey;
	NSString *_widthMetricsKey;
	NSString *_rightMetricsKey;
	NSString *_vertWidthMetricsKey;
	NSString *_topMetricsKey;
	NSString *_bottomMetricsKey;

	BOOL _export;
	NSColor *_color;
	NSInteger _colorIndex;
	NSMutableDictionary *_userData;
	NSDate *_lastChange;
	NSInteger _changeCount;

	NSString *_production;
	BOOL _storeScript;
	BOOL _storeCategory;
	BOOL _storeSubCategory;
	BOOL _storeProduction;
	int _masterCompatibilityCheckStatus;
}
/// @name Properties

/** A pointer to the containing GSFont object */
@property (weak, nonatomic, nullable) GSFont *parent;
@property (readonly, nonatomic, nullable) GSFont *font;
@property (assign, nonatomic) NSInteger changeCount;

/// The glyphs name
@property (nonatomic, copy, nullable) NSString *name;
/// The unicode value of the glyph. This is normaly set automaticaly on setting the name
@property (nonatomic, copy, nullable) NSString *unicode;

@property (strong, nonatomic, nullable) NSOrderedSet *unicodes;

- (void)addUnicode:(NSString *)unicode;

- (NSUInteger)countOfUnicodes;

/** A character code of the glyph of 0 if it has no unicode */
@property (nonatomic, readonly) UTF32Char unicodeChar;

/// The glyphs note
@property (nonatomic, copy, nullable) NSString *note;

/** The language/script of the glyph.

 e.g: latin, cyrillic, greek
 This is set from the internal database.
 */
@property (nonatomic, copy, nullable) NSString *script;
@property (nonatomic) BOOL storeScript;

/** The category of the glyph

e.g: letter, number, punctuation ...
this is set from the internal database.
*/
@property (nonatomic, copy, nullable) NSString *category;
@property (nonatomic) BOOL storeCategory;

/** The sub category of the glyph

e.g: lower case, smallcaps ...
this is set from the internal database.
*/
@property (nonatomic, copy, nullable) NSString *subCategory;
@property (nonatomic) BOOL storeSubCategory;

/** the production name
 */
@property (nonatomic) BOOL storeProduction;
@property (strong, nonatomic, nullable) NSString *production;

/// The glyphs the glyph is composed of.
@property (nonatomic, strong, nullable) NSArray<GSGlyphInfo *> *baseGlyphs;
#ifndef GLYPHS_VIEWER
/** The content of the glyph as dictionary to store it in a pList.

This is used to store the data in the .glyphs file.
*/
@property (unsafe_unretained, readonly) NSDictionary *glyphDict;
/// The glyphname of the keyglyph for LSB.
@property (copy, nonatomic, nullable) NSString *leftMetricsKey;
@property (copy, nonatomic, nullable) NSString *widthMetricsKey;
@property (copy, nonatomic, nullable) NSString *vertWidthMetricsKey;
/// The glyphname of the keyglyph for RSB.
@property (copy, nonatomic, nullable) NSString *rightMetricsKey;
@property (copy, nonatomic, nullable) NSString *topMetricsKey;
@property (copy, nonatomic, nullable) NSString *bottomMetricsKey;
#endif
/// The name of the kerning group for left side kerning.
@property (copy, nonatomic, nullable) NSString *leftKerningGroup;
/// The name of the kerning group for right side kerning.
@property (copy, nonatomic, nullable) NSString *rightKerningGroup;

/** The id of the keyglyph for left side kerning.

this sould look like this: `@MM_L_Name`
*/
@property (copy, nonatomic, nullable) NSString *leftKerningGroupId;

/** The id of the key glyph for right side kerning.

this sould look like this: `@MM_R_Name`
*/
@property (copy, nonatomic, nullable) NSString *rightKerningGroupId;

@property (copy, nonatomic, nullable) NSString *topKerningGroup;

@property (copy, nonatomic, nullable) NSString *bottomKerningGroup;

/** The id of the key glyph for top kerning.

this sould look like this: `@MM_T_Name`
*/
@property (copy, nonatomic, nullable) NSString *topKerningGroupId;

/** The id of the key glyph for bottom kerning.

this sould look like this: `@MM_B_Name`
*/
@property (copy, nonatomic, nullable) NSString *bottomKerningGroupId;

- (BOOL)validateKerningGroup:(id __nullable *__nullable)ioValue error:(NSError *__autoreleasing *)outError;

#ifndef __cplusplus //RMX complains about that.

/// If the glyph will end up in the final font
@property (nonatomic) BOOL export;
#endif

/// The NSColor object of the color label
@property (retain, nonatomic) NSColor *_Nullable color;

/** Returns the index of the color label

@return 0–11, anything else means no label
*/
@property (assign, nonatomic) NSInteger colorIndex;

/** The glyphs undoManager

Each GSGlyph object has its own undoManager

@see [GSFont.undoManager]([GSFont undoManager])

@return a NSUndoManager object
*/

#ifndef GLYPHS_VIEWER
@property (strong, nonatomic, nullable) NSUndoManager *undoManager;
#endif
@property (nonatomic) NSUInteger glyphId;

/** The glyph key is used to access the glyph.

This is used to keep track of the glyph even if is renamed.
It is used for the kerning.
It could look like this: "FBCA074D-FCF3-427E-A700-7E318A949AE5"
*/
@property (strong, nonatomic) NSString *id;
/**
The names is used for sorting.

If set it will be used for sorting inside of
*/
@property (strong, nonatomic, nullable) NSString *sortName;
@property (strong, nonatomic, nullable) NSString *sortNameKeep;

#ifndef LIBCORE
@property (nonatomic, strong, nullable) NSDate *lastChange;
#else
@property (nonatomic, strong, readonly) NSDate *lastChange;
#endif

// needed for control chached interpolation stuff. lasteChange is reset on undo so it can’t cover this.
@property (nonatomic) NSTimeInterval lastOperationInterval;

// for compatibilty with plugins
@property (nonatomic, strong, nullable) NSDate *lastOperation;

#ifndef GLYPHS_LITE
/** Defines the parameters for Smart Glyphs.
 
 It contains an array of GSPartProperty objects.
 */
@property (strong, nonatomic, nullable) NSMutableArray <GSPartProperty *> *partsSettings;

- (NSUInteger)countOfPartsSettings;
- (GSPartProperty *_Nullable)objectInPartsSettingsAtIndex:(NSUInteger)idx;
- (void)insertObject:(GSPartProperty *)partProperty inPartsSettingsAtIndex:(NSUInteger)idx;
//- (void)insertPartsSettings:(NSArray *)_partsSettingArray atIndexes:(NSIndexSet *)indexes;
- (void)removeObjectFromPartsSettingsAtIndex:(NSUInteger)idx;
//- (void)removePartsSettingsAtIndexes:(NSIndexSet *)indexes;
- (void)replaceObjectInPartsSettingsAtIndex:(NSUInteger)idx withObject:(GSPartProperty *)partProperty;

#endif

- (NSString *)charString;

//@property(retain) id itemView;
/// Initialisation
/// Initializes a newly allocated glyph with this name
//- (id) initWithName:(NSString*) Name ;

/** initialises a glyph with a dictionary loaded from a pList.

@param glyphDict A dictionary
*/
- (instancetype)initWithGlyphDict:(NSDictionary *)glyphDict;

#ifndef GLYPHS_VIEWER

/*!
 @discussionsaves the object to file
 @param File		a File
 @param Compact	a bool indecating if it should use a more compact file format
 @param saveLastChange Disable the 'lastChange' entry in the file
 @param error	on return an error containig the reason for failiour
 @return YES if sucsessful
 */

- (BOOL)saveToFile:(FILE *)File compact:(BOOL)Compact doLastChange:(BOOL)saveLastChange error:(NSError *__autoreleasing *)error;

#endif
/// @name Methods

#ifndef GLYPHS_VIEWER
- (void)updateChangeCount:(NSDocumentChangeType)changeType;

- (NSError *_Nullable)validateName:(NSString *)Name;
#endif

- (void)setNameFast:(NSString *)name;

- (void)setName:(NSString *)Name changeName:(BOOL)ChangeName;

- (void)setName:(NSString *)Name changeName:(BOOL)ChangeName update:(BOOL)Update;

#pragma Layers

/// @name Layers

/** An array of GSLayer objects.
 
 It should have at least the same count than font.fontMasters.
 
 The keys are the id of the font masters and should be the same as the [GSLayer.associatedMasterId]([GSLayer associatedMasterId]) if the layer that is associated by the key. For 'master layers', it should also be equal to the [GSLayer.layerId]([GSLayer layerId]).
 */
@property (strong, nonatomic) NSDictionary *layers;

/** The number of layers in the glyph
 
 @return The number of layers
 */
- (NSUInteger)countOfLayers;

/** Returns the a layer loacated at a Index.

 @param Key The Id of the layer
 @return A GSLayer object or nil if not found
*/
- (GSLayer *_Nullable)layerForKey:(NSString *_Nonnull)Key;

- (GSLayer *_Nullable)layerForName:(NSString *_Nonnull)Key;

- (GSLayer *_Nullable)layerForName:(NSString *_Nonnull)Name masterID:(NSString *_Nonnull)masterID;

- (NSArray *_Nullable)layersForName:(NSString *)Name masterIDs:(NSArray *)masterIDs;
/**
 Adds the Layer with the key.
 @discussion If the Key is a id of a GSFontMaster, it will be a masterlayer, otherwise the associated master ID of the layer should be set to a ID of a master.
 @param Layer The GSLayer object.
 @param Key The layer key. For master layers this is the id of the corresponding GSFontMaster.
*/
- (void)setLayer:(GSLayer *)Layer forKey:(id)Key;

- (NSArray *)sortedLayers;
#ifndef GLYPHS_VIEWER

/**
 Removes the Layer for the key.

 @param Key The layer key.
*/
- (void)removeLayerForKey:(id)Key;

#ifndef GLYPHS_LITE
/**
 Returns if the glyphs has compatibel outlines.

 @return YES if the layers have compatible outlines.
*/
- (BOOL)mastersCompatible;

/**
 Returns if the glyphs has compatibel outlines in the layers for all FontMasterIds
 
 @param FontMasterIds the Master IDs
 @return YES if the layers have compatible outlines.
 */
- (BOOL)mastersCompatibleForLayerIds:(NSArray *)FontMasterIds;

- (BOOL)mastersCompatibleForLayers:(NSArray <GSLayer *> *)layers;

- (void)sortLayers:(NSMutableArray *)Layers;

- (NSArray * _Nonnull)sortedBracketLayers;

- (NSArray *)layerGroups:(NSArray *)Instances masters:(NSArray *)Masters error:(NSError *__autoreleasing *)error;

- (NSMutableArray *)forcedLayerGroupIdsMasters:(NSArray * _Nonnull)Masters layers:(NSArray *)layers seenLayers:(NSMutableSet * _Nullable)seenLayers;

- (GSLayer *_Nullable)interpolate:(NSDictionary *)Interpolation NS_DEPRECATED_MAC(10_0, 10_0);

- (GSLayer *)interpolate:(NSDictionary *)Interpolation error:(NSError *__autoreleasing *)error;

/**
 Generates an interpolated layer
 
 @param instance  The instance that defines the interpolation values
 @param keepSmart If smart components sould be decomposed before interpolation.
 @param error if there is a problem, it returnes an NSError object (by reference) that contains details.
 @return The intepolated layer, or nil if something went wrong.
 */
- (GSLayer *)interpolate:(GSInstance *)instance keepSmart:(BOOL)keepSmart error:(NSError *__autoreleasing *)error;

- (GSLayer *)interpolate:(GSInstance *)instance masters:(NSArray *)masters keepSmart:(BOOL)keepSmart error:(NSError *__autoreleasing *)error;

- (GSLayer *_Nullable)interpolate:(GSInstance *_Nullable)Instance masters:(NSArray *)Masters keepSmart:(BOOL)keepSmart smartSettings:(NSDictionary *_Nullable)smartSettings layerName:(NSString *_Nullable)layerName error:(NSError *__autoreleasing *_Nullable)error;

- (GSLayer *_Nullable)interpolate:(GSInstance *_Nullable)Instance masters:(NSArray *)Masters keepSmart:(BOOL)keepSmart smartSettings:(NSDictionary *_Nullable)smartSettings layerName:(NSString *_Nullable)layerName additionalLayers:(NSMutableArray *_Nullable)additionalLayers error:(NSError *__autoreleasing *_Nullable)error;

- (void)interpolateSmartLayer:(GSLayer *)baseLayer instance:(GSInstance *)instance targetGlyph:(GSGlyph *)targetGlyph;
/**
 Replaces the content of the Layer with an interpolation from the remaining layers.
 
 This is useful for 'brace' layers but also for a default state for middle masters.
 @param Layer The Layer that should be recalculated
 */
- (void)replaceLayerWithInterpolation:(GSLayer *)Layer;

#endif
/// Is called from other object after changing some of the glyphs properties.
- (void)outlineHasChanges;

- (id)copyThin:(BOOL)Thin options:(GSGlyphCopyOptions)options;
/**
 If the glyph has MS color glyph information
 */
- (BOOL)isColorGlyph;

- (BOOL)isSVGColorGlyph;
/** If the glyph has Apple color glyph information */
- (BOOL)isAppleColorGlyph;

/** If it is a smart glyph */
- (BOOL)isSmartGlyph;

/** If the glyph has brace or bracket layers */
- (BOOL)hasSpecialLayers;

+ (NSString *_Nullable)modelGlyphForGlyph:(UTF32Char)unicode groups:(NSDictionary *)groups position:(NSString *)groupKey;

#endif

@end
NS_ASSUME_NONNULL_END
