//
//  GSGlyphViewProtocol.h
//  GlyphsCore
//
//  Created by Georg Seifert on 26.01.10.
//  Copyright 2010 schriftgestaltung.de. All rights reserved.
//

#import "GSGeometrieHelper.h"
#import <Cocoa/Cocoa.h>

typedef NS_ENUM(uint8_t, GSKerningMode) {
	GSKerningNormal = 1,
	GSKerningContextual = 2
};

@class GSLayer;
@class GSTextStorage;
@class GSLayoutManager;

NS_ASSUME_NONNULL_BEGIN

/** This protocol defines the API for the graphicView */
@protocol GSGlyphEditViewProtocol

/** The active layer
 
 If only one layer is selected it retunes it, otherwise it retunes nil.
 
 In the editview, this is the layer that has a gray info box.
 */
@property (nonatomic, nullable, strong) GSLayer *activeLayer;

/** The position of the activeLayer
 
 This is in scaled view coordinates.
 
 @see scale
 */
@property (nonatomic) NSPoint activePosition;

- (NSAffineTransform *)zoomTransformerForPosition:(NSPoint)position doOffset:(BOOL)offset;

- (void)bezierPathToViewSpace:(NSBezierPath *)bezierPath atPoint:(NSPoint)Position doOffset:(BOOL)offset;

- (void)bezierPathToViewSpace:(NSBezierPath *)bezierPath atPoint:(NSPoint)Position;

/// The scale of the view
@property (nonatomic) CGFloat scale;

/// The index of the character of the active layer in the string
@property (nonatomic) NSUInteger activeIndex;

/** number of layers in the view.
 
 This might be difference than the number of characters in the string after the features are applied
 */
@property (nonatomic, readonly) NSUInteger cachedLayerCount;

/// the selection range in characters as in the original string
@property (nonatomic) NSRange selectedRange;

/**
 The layer selection range after features have been applied.
 
 @note This should now return the contents of the new `selectedLayerRange` property.
 */
@property (nonatomic, readonly) NSRange cachedLayerSelectionRange;

/**
 The layer selection range after features have been applied.
 
 When setting this property, the `selectedRange` property should also be set to the corresponding converted character range.
 When setting the `selectedRange` property (except in the case above), this property should be set to {NSNotFound, 0}.
 When getting this property, the actual ivar should only be returned if it is not {NSNotFound, 0}. Otherwise, the corresponding converted character range should be returned.
 
 @note Setting this property should implicitly set `selectionAnchorLayerIndex` if its length is 0.
 */
@property (nonatomic) NSRange selectedLayerRange;

/**
 Marks the starting point of selection.
 
 @note This property should be set implicitly when `selectedLayerRange` is set and its length is 0.
 */
@property (nonatomic) NSUInteger selectionAnchorLayerIndex;

/// the currently active cursor
@property (nonatomic, retain, nullable) NSCursor *cursor;

@property (nonatomic) NSRect glyphFrame;

/** the text storage */
@property (strong, nonatomic, nonnull) GSTextStorage *textStorage;

@property (nonatomic) GSKerningMode kerningMode;

/// Set if kerning is active
@property (nonatomic, assign) BOOL doKerning;

/// Set if spacing is active
@property (nonatomic, assign) BOOL doSpacing;

/// The layout manager
@property (strong, nonatomic, nonnull) GSLayoutManager *layoutManager;

@property (readonly, nonatomic, nullable) NSColor *backgroundColor;
@property (readonly, nonatomic, nullable) NSColor *backgroundStrokeColor;
@property (readonly, nonatomic, nullable) NSColor *otherLayersStrokeColor;
@property (readonly, nonatomic, nullable) NSColor *foregroundColor;
@property (readonly, nonatomic, nullable) NSColor *foregroundStrokeColor;
@property (nonatomic) BOOL drawDark;
///

- (NSPoint)getLocation:(NSEvent *_Nullable)theEvent;

/** Translates the view coordinate from the event into glyph coordinates of the active layer
 
 @param theEvent a mouse down even
 
 @return The coordinates
 */
- (NSPoint)getActiveLocation:(NSEvent *_Nullable)theEvent;

/** Returns the cached layer at index
 
 @param Index The cached index
 
 @return The layer
 */
- (GSLayer *_Nullable)cachedGlyphAtIndex:(NSUInteger)Index;

/** Returns the cached position at a layer index
 
 @param Index The cached index
 
 @return the position
 */
- (NSPoint)cachedPositionAtIndex:(NSUInteger)Index;

/**
 Should pass the given graphical position to `-layerIndexForPoint:` and convert the layer index to a character index.

 @note This may not be the desired index if a character is comprised of multiple glyphs. Instead, use `-layerIndexForPoint:` directly and set the `selectedLayerRange` property.

 @param thePoint The graphical position to check, in view coordinates.

 @return The index of the character whose bounding box contains the point.
 */
- (NSUInteger)characterIndexForPoint:(NSPoint)thePoint;

/**
 Converts the given graphical position to a layer index.

 @param thePoint The graphical position to check, in view coordinates.

 @return The index of the layer whose bounding box contains the point.
 */
- (NSUInteger)layerIndexForPoint:(NSPoint)thePoint;

- (void)drawLayer:(GSLayer *_Nonnull)Layer atPoint:(NSPoint)aPoint asActive:(BOOL)Active attributes:(NSDictionary *_Nullable)Attributes;

- (void)drawLayerOutlines:(GSLayer *_Nullable)Layer aPoint:(NSPoint)Position color:(NSColor *_Nullable)Color fill:(BOOL)Fill;

//- (NSDictionary *_Nullable)drawTextAttributes:(NSColor *_Nullable)TextColor NS_DEPRECATED_MAC(10_0, 10_0);

- (void)reflow;

- (BOOL)addImageFromPasteBoard:(NSPasteboard *_Nonnull)pasteboard;

- (void)replaceCharactersInRange:(NSRange)aRange withString:(nonnull NSString *)aString;

@end

NS_ASSUME_NONNULL_END
