//
//  GSComponent.h
//  Glyphs
//
//  Created by Georg Seifert on 30.3.08.
//  Copyright 2008 schriftgestaltung.de. All rights reserved.
//

#import "GSElement.h"
#import <Cocoa/Cocoa.h>
#import <GlyphsCore/GSUserDataProtocol.h>

NS_ASSUME_NONNULL_BEGIN

@class GSGlyph;
@class GSFont;
@class GSLayer;
@class GSNotifyingDictionary;
@class GSPartProperty;

typedef NS_ENUM(int8_t, GSComponentAlignment) {
	GSAlignmentNoAligned = -2,
	GSAlignmentDisable = -1,
	GSAlignmentDefault = 0,
	GSAlignmentForce = 1,
	GSAlignmentAligned = 2,
	GSAlignmentHorizontal = 3,
};

/// This class defines the component object. It is a subclass of GSElement

@interface GSComponent : GSElement <GSUserDataProtocol> {
	NSString *_componentName;
	GSGlyph *_component;
	GSComponentAlignment _isAligned;
	GSLayer *_cachedLayer;
	NSDate *_lastChange;
  @public
	NSAffineTransformStruct _transformStruct;
	NSString *_anchor;
	// BOOL _disableAlignment;
	GSComponentAlignment _alignment;
	CGFloat _keepWeight;
}
/** components are referenced by the glyph name.

If you set the componentName the component value is set and this is used during runtime. So you can change the glyph name of the base glyph during runtime.
*/
@property (strong, nonatomic) NSString *componentName;

/** The base glyph

This is readonly. To set the base glyph use componentName: newBaseGlyph.name.
@see componentName
*/
@property (unsafe_unretained, readonly) GSGlyph *component;

/** The transformation of the component.

The values of the struct are:
{m11, m12, m21, m22, tX, tY}
this is equivalent to the glif attributes
{xScale, xyScale, yxScale, yScale, xOffset, yOffset}
in the same order.
The initial value is {1,0,0,1,0,0}.
*/
@property (assign, nonatomic) NSAffineTransformStruct transformStruct;

- (void)setTransformStructFast:(NSAffineTransformStruct)newTS round:(CGFloat)Grid;

/** If more than one anchor/_anchor pair would match, this property can be used to set the anchor to use for automatic alignment
 
 This can be set from the anchor button in the component info box in the UI
 */
@property (copy, nonatomic) NSString *anchor;
@property (assign, nonatomic) CGFloat keepWeight;

// @property (nonatomic, assign) BOOL disableAlignment;
@property (nonatomic, assign) GSComponentAlignment alignment;

- (void)setAlignmentFast:(GSComponentAlignment)flag;

@property (nonatomic) GSComponentAlignment isAligned;

//Compatibility
@property (nonatomic, readonly) BOOL disableAlignment;

@property (nonatomic, readonly) BOOL doesAlign;
@property (nonatomic, readonly) BOOL shouldAlign;
@property (nonatomic, readonly) BOOL doesAttach;
@property (nonatomic, readonly) BOOL shouldAttach;

@property (nonatomic, readonly) GSComponentAlignment alignmentType;
#ifndef GLYPHS_LITE
/** The instance settings do Smart Components. 
 
 It is a dictionary containing the parameter names as keys and the interpolation position as value.

	{
		Height = 70;
		Width = 70;
	}
 
 @see GSGlyph.partsSettings
 */
@property (strong, nonatomic) GSNotifyingDictionary *pieceSettings;

- (void)setPieceSettingsFast:(GSNotifyingDictionary *)pieceSettings;

- (void)setPieceValue:(CGFloat)value forKey:(NSString *)key;

- (CGFloat)pieceValueForKey:(NSString *)key;

- (NSString *_Nullable)hangulModelGlyph;

- (BOOL)isHangulKeyComponent;

- (void)convertPieceSettingsToID;
#endif
/** The bounds of the components
 
 @return the bounds
 */
- (NSRect)bounds;

/// The bezierPath of the component. It is already transformed.
- (NSBezierPath *_Nullable)bezierPath;

- (NSBezierPath *)bezierPathIgnorePlaceholder:(BOOL)ignorePlaceholder;

/// The bezierPath of open paths of the component. It is already transformed.
- (NSBezierPath *_Nullable)openBezierPath;

- (NSData *)bezierData;

- (NSTimeInterval)lastOperation:(NSMutableSet *)seenComponents;

- (void)updateComponentGlyph;

- (void)updateComponentGlyphDontNotifiy;
- (void)updateComponentGlyphFast;
/** The Layer that the component is linked to.
 
 The is usually just the layer from the components glyph but might be the computed result of a smart component.
 
 @return the Layer that represents the component.
 */
- (GSLayer *)componentLayer;

/** Retunes on reference the scale and rotation of the component
 
 This will take apart the transformStruct.
 
 This might not work correctly if you have applied a skew transform.
 
 @param sX On return, the horizontal scale
 @param sY On return, the vertical scale
 @param R  On return, the rotation
 */
- (void)getScaleX:(CGFloat *)sX scaleY:(CGFloat *)sY rotation:(CGFloat *)R;

#ifndef GLYPHS_VIEWER

/** Sets the transformation.
 
 @param sX the horizontal scale
 @param sY the vertical scale
 @param R  the rotation
 */
- (void)setScaleX:(CGFloat)sX scaleY:(CGFloat)sY rotation:(CGFloat)R;

/** Decomposes the component to the containing Layer

It need to parent property set correctly. It calls `[_parent decomposeComponent:self]`.
*/
- (void)decompose;

#ifndef GLYPHS_LITE

/** If the component overlaps with the OtherComponent
 
 @param OtherComponent The OtherComponent to test overlap.

 @return Returns YES if both components do overlap, otherwise NO.
 */
- (BOOL)intersects:(GSComponent *)OtherComponent;

#endif

/** It makes the component the first in the containing Layer

It need to parent property set correctly. It calls `[_parent makeFirstComponent:self]`.
*/
- (void)makeFirst;

- (void)makeDisableAlignment;

- (void)makeEnableAlignment;

- (void)makeForceAlignment;

- (CGFloat)valueForAxis:(GSPartProperty *)Axis;

- (void)preloadCachedLayers;
#endif

@end
NS_ASSUME_NONNULL_END
