//
//  GSFont.h
//  Glyphs
//
//  Created by Georg Seifert on 21.10.05.
//  Copyright 2005 schriftgestaltung.de. All rights reserved.
//
#import <Cocoa/Cocoa.h>

@class GSGlyph;
@class GSLayer;
@class GSDocument;
@class GSFontMaster;
@class GSInstance;
@class GSFeature;
@class GSClass;
@class GSFeaturePrefix;
@class COFont;
@class MGOrderedDictionary;
@class GSCustomParameter;
@class GSGlyphsInfo;

#import <GlyphsCore/GSContainerProtocol.h>
#import <GlyphsCore/GSUserDataProtocol.h>

NS_ASSUME_NONNULL_BEGIN

/** Used to determine how to gemerate the numbers features (pnum, lnum) */
typedef NS_ENUM(uint8_t, GSFigureStyles) {
	/** This is the initial state*/
	NotTested = 0,
	/** There are no figures */
	NoFigures = 1,
	/** The default figures are tabular oldstyle figures */
	tosf = 2,
	/** The default figures are proportional oldstyle figures */
	osf = 3,
	/** The default figures are tabular figures */
	tf = 4,
	/** The default figures are lining figures */
	lf = 5
};

/**
 Possible writing directions
 */
typedef NS_ENUM(int8_t, GSWritingDirection) {
	/**
	 a default value, not used
	 */
	GSWritingDirectionNatural = -1,
	/**
	 Left to Right
	 */
	GSWritingDirectionLeftToRight = 0,
	/**
	 Right to Left
	 */
	GSWritingDirectionRightToLeft = 1,
	/**
	 Vertical to Left
	 */
	GSWritingDirectionVerticalToLeft = 2,
	/**
	 Vertical to Right
	 */
	GSWritingDirectionVerticalToRight = 3,
};

NSInteger compareKerning(NSString *obj1, NSString *obj2, void *context);
/** This class defines the font object.

 Font info properties, that have a user interface, have proper accessors, eveything else is stored in the custom parameters
 */

@interface GSFont : NSObject <NSCoding, NSCopying, GSContainerProtocol, GSUserDataProtocol> {
	GSDocument *__weak _parent;
	NSMutableArray<GSGlyph *> *_glyphs;

	NSString *_familyName;
	NSUInteger _unitsPerEm;
	NSDate *_date;
	NSUInteger _versionMajor;
	NSUInteger _versionMinor;
	NSString *_designer, *_designerURL, *_manufacturer, *_manufacturerURL, *_copyright;

	NSMutableArray *_customParameters;
	NSMutableArray *_fontMasters;
	NSMutableArray *_features;
	NSMutableArray *_instances;
	NSMutableDictionary *_userData;
	NSMutableDictionary *_tempData;
	NSMutableDictionary *_charsCache;
	UTF32Char _lastChar;
	MGOrderedDictionary *_kerning;
	MGOrderedDictionary *_contextKerning;
	MGOrderedDictionary *_verticalKerning;
	NSUInteger _gridMain;
	NSUInteger _gridSubDivision;
	CGFloat _keyboardIncrement;
	BOOL _disablesNiceNames;
	BOOL _disablesAutomaticAlignment;
	BOOL _keepAlternatesTogether;
	NSMutableDictionary *name2Glyph;
	NSDictionary *id2Glyph;
	NSMutableArray *GlyphDictArray;
	NSMutableArray *_classes;
	NSMutableArray *_featurePrefixes;
	NSString *_tempOTFFont;
	COFont *_compositorFont;
	BOOL _disabledChanges;
	NSMutableSet *_disabledChangeKeys;
	NSDate *_lastAskedForImage;
	BOOL _hasRegisteredObservers;
}
//! The pointer to the document object
@property (weak, nonatomic, nullable) GSDocument *parent;

/** The undoManager of the document
 
 This is only used for font info and for adding/removing glyphs. Each glyph has it’s own undoManager.
 
 @see [GSGlyph.undoManager]([GSGlyph undoManager])
 */
@property (nonatomic, readonly, nullable) NSUndoManager *undoManager;

/** Returns the content of the font object to store it in pList.

 This is used to store the data in the .glyphs file.
 */
@property (unsafe_unretained, readonly, nonatomic) NSMutableDictionary *fontDict;

#pragma mark Info
/** @name Font Info */

//! The (family) name of the font.
@property (strong, nonatomic) NSString *familyName;

//! A note
@property (strong, nonatomic, nullable) NSString *note;

/// unitsPerEm (UPM).
@property (assign, nonatomic) NSUInteger unitsPerEm;

/// copyright
@property (strong, nonatomic, nullable) NSString *copyright;

/// designer
@property (strong, nonatomic, nullable) NSString *designer;

/// designerURL
@property (strong, nonatomic, nullable) NSString *designerURL;

/// manufacturer
@property (strong, nonatomic, nullable) NSString *manufacturer;

/// manufacturerURL
@property (strong, nonatomic, nullable) NSString *manufacturerURL;

/// versionMajor
@property (assign, nonatomic) NSUInteger versionMajor;

/// versionMinor
@property (assign, nonatomic) NSUInteger versionMinor;

/// The font date
@property (strong, nonatomic) NSDate *date;


#pragma mark tempData
/**
 a mutable dictionary that stores data. It will not be written to disk.
 */
@property (nonatomic, strong, nullable) NSMutableDictionary *tempData;

/**
 Adds key/value to tempData. Pass nil as value to remove previous set data

 @param value and object or nil
 @param key the key
 */
- (void)setTempData:(nullable id)value forKey:(nonnull NSString *)key;

/** disablesNiceNames

 This is to prevent the automatic glyph name substitution.
 Set it to YES before you do big changes. Make sure to disable it after you are finished.
 */
@property BOOL disablesNiceNames;

/** Disables automatic alignment of components */
@property (nonatomic) BOOL disablesAutomaticAlignment;

/** Keeps alternated glyphs next to the base glyph */
@property (nonatomic) BOOL keepAlternatesTogether;

@property (nonatomic, strong, nullable) NSMutableSet *disabledChangeKeys;

#pragma mark -
/** @name Creating and Initializing a GSFont object */

/** initializes a font with a dictionary loaded from a pList.

 @param fontDict A dictionary
 */
- (instancetype)initWithFontDict:(NSDictionary *)fontDict;

/**
 Initializes a font with the contents of the location specified by a given URL.

 @param fileURL the location of the file
 @param outError If there is a error it contains an NSError object
 @return the GSFont object or nil
 */
- (instancetype)initWithURL:(NSURL *)fileURL error:(NSError **)outError;

- (void)postProcess;

/** @name Font Methods */

#ifndef GLYPHS_VIEWER
#ifndef LIBCORE

/** Scale the hole font.

 @param Scale Multiplier for scale the font (glyphs, kerning and metrics)
 */
- (void)scaleBy:(CGFloat)Scale;

#endif
#endif
#pragma mark glyphs

/** @name Axes */

@property (readonly) NSArray *axes;
- (NSArray *)legacyAxes;
+ (int)getAxisCountFromMasters:(NSArray *)masters;

- (NSUInteger)countOfAxes;

/** @name Glyphs */

//! The array of GSGlyph objects
@property (strong, nonatomic) NSArray<GSGlyph *> *glyphs;

- (NSUInteger)count;

/** The count of glyphs in the font */
- (NSUInteger)countOfGlyphs;

/** The glyph object located at index.

 Checks for valid index and returns nil if index is to high.
 
 @param Index An index within the bounds of the the glyphs array.
 @return a GSGlyph object.
 */
- (GSGlyph *)glyphAtIndex:(NSInteger)Index;

/** Returns the index of the glyph.
 
 @param Glyph The Glyph you are looking for
 @return The Index of the glyph or NSNotFound if not found
 */
- (NSInteger)indexOfGlyph:(GSGlyph *)Glyph;

@property (nonatomic, readonly) NSDictionary *charsCache;

/** If you changed the glyphs array yourself, call that method to reset some chances. */
- (void)resetCache;

/** Returns the GSGlyph with this name.

 @warning This is a bit slow.

 @param Name the name of the glyph.
 @return a GSGlyph object.
 */
- (GSGlyph *)glyphForName:(NSString *)Name;

#ifndef LIBCORE
#ifndef GLYPHS_LITE
- (NSArray *)glyphNamesForWildcard:(NSString *)name;
#endif

/**
 A list of glyph names matching query

 @param query This can be a name containing wildcards ("a.*") or a query ("script=arabic")
 @param error if something goes wrong, it contains an error on return
 @return a list of glyph names or nil
 */
- (NSArray *)glyphNamesForWildcard:(NSString *)query error:(NSError *__autoreleasing *)error;

- (NSString *)baseName:(NSString *)name forWildCard:(NSString *)wild;
- (void)prepareName2Glyph;
#endif
- (GSGlyph *)glyphForNameFast:(NSString *)Name;


/** A list of glyph names

 @return A list of glyph names
 */
- (NSArray *)glyphNames;


/** Returns the GSGlyph with this this Unicode.
 
 @param Unicode the unicode as a hex string ('004C')
 @return a GSGlyph object or nil if not found.
 @see resetCache If you changed some glyphs and this method returns wrong results, call resetCache
 */
- (GSGlyph *)glyphForUnicode:(NSString *)Unicode;


/** Returns the GSGlyph with this this Unicode.
 
 @param unicodes a list of unicodes as hex strings ('004C')
 @return a GSGlyph object or nil if not found.
 @see resetCache If you changed some glyphs and this method returns wrong results, call resetCache
 */
- (GSGlyph *)glyphForUnicodes:(NSOrderedSet *)unicodes;

- (void)glyphHasChangedName:(GSGlyph *)Glyph oldName:(NSString *)OldName;

#ifndef GLYPHS_VIEWER
- (GSGlyph *)newGlyphWithName:(NSString *)glyphName;

/** Adds a new glyph with this glyphName to the font
 
 @param glyphName  The new glyph name
 @param changeName If changeName is yes, the glyphName will be converted to nice-name, otherwise not.
 
 @return The new glyph.
 */
- (GSGlyph *)newGlyphWithName:(NSString *)glyphName changeName:(BOOL)changeName;

- (void)makeSaveGlyphInfo:(GSGlyph *)glyph;

/** Calculates a names that is unique for the font
 
 @param Glyph The Glyph
 
 @return The glyph name with a .0XX suffix if the name is already in use by another glyph
 */
- (NSString *)saveNameForGlyph:(GSGlyph *)Glyph;

- (NSString *)saveSuffixForGlyph:(GSGlyph *)Glyph;
/** Calculates a names that is unique for the font
 
 @param GlyphName The glyph name
 
 @return The glyph name with a .0XX suffix if the name is already in use by another glyph
 */
- (NSString *)saveNameForName:(NSString *)GlyphName;

#endif

/** Returns the GSGlyph with this ID.

 @warning This is a bit slow.

 @param Id the ID of the glyph.
 @return a GSGlyph object.
 */
- (GSGlyph *)glyphForId:(NSString *)Id;

- (void)setDefaultWidth:(GSGlyph *)glyph;

#ifndef GLYPHS_VIEWER
/** Tests if all the glyphs are present and set to export
 
 @param GlyphNames A array of glyph names
 @return YES if all glyphs will end up in a exported font.
 @see hasGlyphs:
 @see filterKeepGlyphs:
 @see filterGlyphs:
 */
- (BOOL)keepsGlyphs:(NSArray<NSString *> *)GlyphNames;

/** Returns a Boolean value if the font contains the glyphs
 
 @param GlyphNames A array of glyph names
 @return YES if all glyphs are in the font.
 @see keepsGlyphs:
 @see filterKeepGlyphs:
 @see filterGlyphs:

 */
- (BOOL)hasGlyphs:(NSArray *)GlyphNames;

#endif
/** Returns all glyph names that are present and set to export
 
 @param GlyphNames A array of glyph names
 @return A list of exporting glyphs
 @see keepsGlyphs:
 @see hasGlyphs:
 @see filterGlyphs:
 */
- (NSArray *_Nullable)filterKeepGlyphs:(NSArray *)GlyphNames;

/** Returns all glyph names that are present and set to export
 
 @param GlyphNames A array of glyph names
 @return A list of all glyphs that are in the font
 @see keepsGlyphs:
 @see hasGlyphs:
 @see filterKeepGlyphs:
 */
- (NSArray *_Nullable)filterGlyphs:(NSArray *)GlyphNames;

- (NSArray *)glyphsContainingComponentWithName:(NSString *)GlyphName masterID:(NSString *)MasterID;
/** Returns the glyph with the chars unicode.

 @param Char A (unicode) char.
 @return a GSGlyph object
 @see characterForGlyph:
 */
- (GSGlyph *)glyphForCharacter:(UTF32Char)Char;

/** Returns a UTF32Char for the glyph.

 It uses the unicode to find the char value. For unencoded glyphs it returns a cached PUA value.

 @param Glyph a GSGlyph object
 @return a UTF32Char
 @sa glyphForCharacter:
 */
- (UTF32Char)characterForGlyph:(GSGlyph *)Glyph;

#ifndef GLYPHS_VIEWER

- (NSSet *)componentGlyphNamesForGlyph:(GSGlyph *)Glyph;

/** Adds the Glyph object to the font.

 @param Glyph A GSGlyph object.
 */
- (void)addGlyph:(GSGlyph *)Glyph;

/** Adds the Glyphs object to the font.

 @param Glyphs A NSArray of GSGlyph objects.
 */
- (void)addGlyphsFromArray:(NSArray *)Glyphs;

/** Removes the Glyph from the font

 @param Glyph A GSGlyph to remove from the font.
 */
- (void)removeGlyph:(GSGlyph *)Glyph;

/** Removes the Glyphs from the font

 @param Glyphs A NSArray of GSGlyph to remove from the font.
 */
- (void)removeGlyphs:(NSArray *)Glyphs;

#endif

/** Is called if the there are changes. */
- (void)sortGlyphs;


/** Triggers the soring of kerning */
- (void)sortKerning;


/**
 This is called be child objects to tell the containing class that it has changed

 @param element the calling object
 */
- (void)elementDidChange:(id)element;

- (void)notifyUpdate;

#pragma mark fontInfoDict accessors

/** @name Custom Parameters */

/** List of custom parameters */
@property (nonatomic, retain) NSMutableArray *customParameters;

/** Returns the custom property with Name
 
 This will always return the first entry with this name. If there are multiple parameters with the same name, parse the customParameters array directly.
 
 @param key The name of the parameter
 
 @return The parameter object
 @see customValueForKey:
 */
- (GSCustomParameter *)customParameterForKey:(NSString *)key;

/** The number of parameters in the font */
- (NSUInteger)countOfCustomParameters;

/** This returns the value of the parameter object with Name
 
 This will always return the first entry with this name. If there are multiple parameters with the same name, parse the customParameters array directly.
 
 @param key The name of the parameter
 
 @return The value of the parameter object object
 @see customParameterForKey:
 */
- (id)customValueForKey:(NSString *)key;



/** A convenient method to check if a custom parameter is set and has a positiv setting

 @param key the name of the custom parameter
 @return YES/NO
 */
- (BOOL)customBoolValueForKey:(NSString *)key;

/** Sets the key value pair in the font
 
 if a parameter with this name is already present, its value will be updated. Otherwise a new parameter will be added
 
 @param value a value
 @param key   The name of the parameter
 */
- (void)setCustomParameter:(id)value forKey:(NSString *)key;


/** adds a parameter to the list font

 @param Property a GSCustomParameter
 */
- (void)addCustomParameter:(GSCustomParameter *)Property;

/** Removes the parameter with this name from the font

 It will only remove the first parameter with that name
 
 @param key   The name of the parameter

 */
- (void)removeObjectFromCustomParametersForKey:(id)key;

/** Removes the parameter a theIndex 
 
 @param index The index
 */
- (void)removeObjectFromCustomParametersAtIndex:(NSUInteger)index;

/** The parameter at theIndex 
 
 @param index The index
 @return The GSCustomParameter or nil of index is out of range
 */
- (id)objectInCustomParametersAtIndex:(NSUInteger)index;

/** The panose value for the font */
- (NSMutableArray *)panose;

#pragma mark masters
/** @name Masters */

//! An array of GSFontMaster objects
@property (strong, nonatomic) NSArray *fontMasters;

/** The count of masters in the font.

 @return The number of layers currently in the font.
 */
- (NSInteger)countOfFontMasters;

/** The master located at index

 @param Index An index
 @return A GSFontMaster instance or nil if index is out of bounds
 */
- (GSFontMaster *)fontMasterAtIndex:(NSInteger)Index;

/** Returns the fontmaster with this id

 @param Id The ID of the font master. This is supposed to be a Unique id
 @return A GSFontMaster instance.
 */
- (GSFontMaster *)fontMasterForId:(NSString *)Id;

/** Inserts the FontMaster at Index
 
 @param FontMaster The master to add to the font
 @param theIndex   The index to insert FontMaster.
 */
- (void)insertFontMaster:(GSFontMaster *)FontMaster atIndex:(NSUInteger)theIndex;

/** Adds the fontMaster to the font
 
 @param fontMaster The master to add.
 */
- (void)addFontMaster:(GSFontMaster *)fontMaster;

/** Removes the fontMaster from the font
 
 @param fontMaster The master to remove.
 */
- (void)removeFontMaster:(GSFontMaster *)fontMaster;
- (void)removeFontMasterAndContent:(GSFontMaster *)FontMaster;

- (void)replaceFontMasterAtIndex:(NSUInteger)theIndex withFontMaster:(GSFontMaster *)FontMaster;

/** Removes the fontmaster with this id

 @param Index The index of the FontMaster to remove.
 */
- (void)removeFontMasterAtIndex:(NSInteger)Index;

#ifndef GLYPHS_LITE
- (void)copyInfoFrom:(GSFont *)sourceFont sourceFontMasterID:(NSString *)sourceFontMasterID targetFontMasterID:(NSString *)targetFontMasterID;

- (GSFontMaster *)addFontAsNewMaster:(GSFontMaster *)master;
#endif

#ifndef GLYPHS_VIEWER
/** 
 Recalculate the LSB and RSB for all glyph
 
 This will not apply the metrics keys.
 */
- (void)updateMetrics;

#endif
#pragma mark instances

/** @name Instances */

//! An array of GSInstance objects. This define export settings for interpolation and naming.
@property (strong, nonatomic) NSArray<GSInstance *> *instances;

/** The count of the Instances

 @return The number of Instances currently in the Font.
 */
- (NSUInteger)countOfInstances;

/** The Instance at index

 @param index The index of the instance.
 @return The Instance located at _index_.
 */
- (GSInstance *)objectInInstancesAtIndex:(NSUInteger)index;

#ifndef GLYPHS_VIEWER

/** Inserts the instance at Index

 @param Instance The instance to insert.
 @param index The _index_ at what to insert the _instance_.
 */
- (void)insertObject:(GSInstance *)Instance inInstancesAtIndex:(NSUInteger)index;
- (void)insertInstance:(GSInstance *)Instance atIndex:(NSUInteger)index;
#endif

/** Adds the Instance to the font

 @param Instance The Instance to add to the font.
 */
- (void)addInstance:(GSInstance *)Instance;

#ifndef GLYPHS_VIEWER

- (void)removeInstance:(GSInstance *)Instance;

- (void)removeInstances:(NSArray *)Instances;
/** Removes the Instance at Index.

 @param index The of the Instance to be removed.
 */
- (void)removeObjectFromInstancesAtIndex:(NSUInteger)index;

- (void)replaceObjectInInstancesAtIndex:(NSUInteger)theIndex withObject:(GSInstance *)Instance;

- (GSFontMaster *)interpolateFontMaster:(GSInstance *)Instance interpolation:(NSDictionary *)interpolation scale:(CGFloat)Scale thin:(BOOL)Thin error:(NSError *__autoreleasing *)Error;

/** Generates an instance

 It returns a new GSFont object with one master interpolated all glyphs/layers according to the settings in Instance.

 @param Instance The Instance to use for interpolation.
 @param Error if there is a problem, it returns an NSError object (by reference) that contains details.
 @return The interpolated font, or nil if something went wrong.
 */
- (GSFont *_Nullable)generateInstance:(GSInstance *)Instance error:(NSError *__autoreleasing *)Error;

- (BOOL)applyFilter:(NSArray *)CustomParameters withKey:(NSString *)FilterKey error:(NSError **)error;
#endif
#pragma mark kerning

/** @name Kerning */

- (NSString *)masterIDforMetrics:(NSString *)MasterID;

- (MGOrderedDictionary *)masterKerning:(NSString *)MasterID direction:(GSWritingDirection)direction;

/** the Kerning of the Font
 
 1) A Dict for each FontMaster (FontMaster id as key)
 2) The Dict contains an entry for each glyph (the glyphID or GroupName (@MMK_L_XXX) is used as key)
 3) These entries hold the values: glyphID or GroupName (@MMK_R_XXX) as key, kerning as value.
 */
@property (strong, nonatomic) MGOrderedDictionary *kerning;
@property (strong, nonatomic) MGOrderedDictionary *contextKerning;

/** the vertical Kerning of the Font
 
 1) A Dict for each FontMaster (FontMaster id as key)
 2) The Dict contains an entry for each glyph (the glyphID or GroupName (@MMK_L_XXX) is used as key)
 3) These entries hold the values: glyphID or GroupName (@MMK_R_XXX) as key, kerning as value.
 */
@property (strong, nonatomic) MGOrderedDictionary *verticalKerning;

#ifndef GLYPHS_VIEWER

/** Compresses the kerning

 if a GSGlyph has a Kerning Group and Direct Kerning, the Kerning is changed to Class Kerning.
 */
- (void)compressKerning;

- (void)cleanUpKerning;

#endif

- (CGFloat)kerningForFontMasterID:(NSString *)fontMasterID firstGlyph:(GSGlyph *)firstGlyph secondGlyph:(GSGlyph *)secondGlyph direction:(GSWritingDirection)direction;

- (CGFloat)kerningForFontMasterDict:(NSDictionary *)masterKerning firstGlyph:(GSGlyph *)firstGlyph secondGlyph:(GSGlyph *)secondGlyph direction:(GSWritingDirection)direction;

/** Returns a kerning value

 @param FontMasterID The Master id
 @param LeftKey either a glyph id or a glyph group (@MM_L_XXX)
 @param RightKey either a glyph id or a glyph group (@MM_R_XXX)
 @return the kerning value
 */

- (CGFloat)kerningForFontMasterID:(id)FontMasterID LeftKey:(id)LeftKey RightKey:(id)RightKey direction:(GSWritingDirection)direction;

/** adds a kerning value

 @param FontMasterID The Master id
 @param LeftKey either a glyph id or a glyph group (@MM_L_XXX)
 @param RightKey either a glyph id or a glyph group (@MM_R_XXX)
 @param Value the kerning value
 */

- (void)setKerningForFontMasterID:(id)FontMasterID LeftKey:(id)LeftKey RightKey:(id)RightKey Value:(CGFloat)Value direction:(GSWritingDirection)direction;

/** removes a kerning value

 @param FontMasterID The Master id
 @param LeftKey either a glyph id or a glyph group (@MM_L_XXX)
 @param RightKey either a glyph id or a glyph group (@MM_R_XXX)
 */
- (void)removeKerningForFontMasterID:(NSString *)FontMasterID LeftKey:(NSString *)LeftKey RightKey:(NSString *)RightKey NS_DEPRECATED_MAC(10_0, 10_0);

- (void)removeKerningForFontMasterID:(NSString *)FontMasterID LeftKey:(NSString *)LeftKey RightKey:(NSString *)RightKey direction:(GSWritingDirection)direction;

- (CGFloat)verticalKerningForFontMasterID:(NSString *)FontMasterID LeftKey:(NSString *)LeftKey RightKey:(NSString *)RightKey NS_DEPRECATED_MAC(10_0, 10_0);

- (void)setVerticalKerningForFontMasterID:(NSString *)FontMasterID LeftKey:(NSString *)LeftKey RightKey:(NSString *)RightKey Value:(CGFloat)Value NS_DEPRECATED_MAC(10_0, 10_0);

- (void)removeVerticalKerningForFontMasterID:(NSString *)FontMasterID LeftKey:(NSString *)LeftKey RightKey:(NSString *)RightKey NS_DEPRECATED_MAC(10_0, 10_0);

- (CGFloat)contextKerningForFontMasterID:(NSString *)FontMasterID LeftKey:(NSString *)LeftKey RightKey:(NSString *)RightKey before:(NSString *)BeforeKey after:(NSString *)AfterKey NS_DEPRECATED_MAC(10_0, 10_0);

- (void)setContextKerningForFontMasterID:(NSString *)FontMasterID LeftKey:(NSString *)LeftKey RightKey:(NSString *)RightKey Value:(CGFloat)Value before:(NSString *)BeforeKey after:(NSString *)AfterKey;

- (void)removeContextKerningForFontMasterID:(NSString *)FontMasterID LeftKey:(NSString *)LeftKey RightKey:(NSString *)RightKey before:(NSString *)BeforeKey after:(id)AfterKey;

#pragma mark features

/** @name Features */

//! An array of GSFeature objects.
@property (strong, nonatomic) NSArray *features;

- (NSMutableArray *)mutableFeatures;

/** The count of the features

 @return The number of features in the Font.
 */
- (NSUInteger)countOfFeatures;

/** The feature at index
 
 @param idx The index of the feature
 @return The feature located at theIndex.
 */
- (GSFeature *)objectInFeaturesAtIndex:(NSUInteger)idx;

//! Inserts the Feature at theIndex
- (void)insertObject:(GSFeature *)feature inFeaturesAtIndex:(NSUInteger)idx;

//! Removes the Feature at theIndex
- (void)removeObjectFromFeaturesAtIndex:(NSUInteger)idx;

- (void)replaceObjectInFeaturesAtIndex:(NSUInteger)idx withObject:(GSFeature *)feature;

//- (void)replaceFeaturesAtIndexes:(NSIndexSet *)indexes withFeatures:(NSArray *)featureArray;

//! Adds the Feature
- (void)addFeature:(GSFeature *)Feature;

- (void)removeFeature:(GSFeature *)Feature;

/** Returns the feature with the name or tag.

 @param Tag The feature tag (like "smcp")
 */
- (GSFeature *)featureForTag:(NSString *)Tag;

- (GSFeature *)automaticFeatureForTag:(NSString *)Tag;
#ifndef GLYPHS_VIEWER
#ifndef GLYPHS_LITE

- (BOOL)importFeatures:(NSString *)FeaturesString error:(NSError *__autoreleasing *)error;

#endif
#pragma mark Classes

/** @name Classes */

//! An array of GSClasse objects
@property (strong, nonatomic) NSArray *classes;

- (NSMutableArray *)mutableClasses;

//! The count of the classes
- (NSUInteger)countOfClasses;

- (id)objectInClassesAtIndex:(NSUInteger)theIndex;

/** Returns the class with the name.

 @param Tag The class name.
 */
- (GSClass *)classForTag:(NSString *)Tag;

/** Adds a class to the font
 
 @param theClass the Class
 */
- (void)addClass:(GSClass *)theClass;

/** Insert the Class at theIndex in the font

 @param theClass the class
 @param theIndex the index
 */
- (void)insertObject:(GSClass *)theClass inClassesAtIndex:(NSUInteger)theIndex;

/** Remove the class at theIndex from the font 
 
 @param theIndex the index
 */
- (void)removeObjectFromClassesAtIndex:(NSUInteger)theIndex;

/** Removes the class from the font

 @param theClass the class
 */
- (void)removeClass:(GSClass *)theClass;

/** Replaces the the class a theIndex with Class
 
 @param theIndex the index
 @param theClass the class
 */
- (void)replaceObjectInClassesAtIndex:(NSUInteger)theIndex withObject:(GSClass *)theClass;

/** Convient method to add a class with the Text

 @param Text the class in FDK syntax, e.g. "@myClass = [A B];"
 */
- (void)addClassFromCode:(NSString *)Text;

#pragma mark FeaturePrefixes

/** @name FeaturePrefixes */

/** An array of GSClass objects that contain additional lookups */
@property (nonatomic, strong) NSMutableArray *featurePrefixes;

/** Adds the Prefix to the font 
 
 @param thePrefix the prefix
 */
- (void)addFeaturePrefix:(GSClass *)thePrefix;

#endif
/** the count of the Prefixes in the font */
- (NSUInteger)countOfFeaturePrefixes;

- (GSFeaturePrefix *)objectInFeaturePrefixesAtIndex:(NSUInteger)theIndex;

/** Returns the Prefix with the name.
 
 @param Tag The prefix name.
 */
- (GSClass *)featurePrefixForTag:(NSString *)Tag;

#ifndef GLYPHS_VIEWER
/** Insert the Prefix at theIndex in the font
 
 @param thePrefix the prefix
 @param theIndex the index
 */
- (void)insertObject:(id)thePrefix inFeaturePrefixesAtIndex:(unsigned)theIndex;

/** Remove the Prefix at theIndex from the font

 @param thePrefix the prefix
 */
- (void)removeFeaturePrefix:(GSClass *)thePrefix;

#pragma mark display
/** Converts a string that contains slash seperated glyph names in a regular unicode string

 @param String A String containing slash seperated glyph names
 @return A UTF-16 string. For it will assigne a cached PUA codes for unencoded glyphs
 */
- (NSString *)charStringFromDisplayString:(NSString *)String;

/** Delayes redrawing and updates.
 
 Will record the changes and update once in `enableUpdateInterface`. If you don’t need updates, use `stopUpdateInterface`

 @see stopUpdateInterface
 @warning Always call enableUpdateInterface afterwards
 */
- (void)disableUpdateInterface;

/** Disable redrawing and updates completly.
 
 Changes will not trigger an update. Only use this calefully and call enableUpdateInterface if you need future updates.

 @see disableUpdateInterface
 @warning Always call enableUpdateInterface afterwards
 */
- (void)stopUpdateInterface;

/** Reanables redrawing and updates.
 
 This only has an effect if disableUpdateInterface was called before
 */
- (void)enableUpdateInterface;

- (void)enableFutureUpdates;
/** Check if the updating is enabled
 
 @return NO if disableUpdateInterface was called
 */
- (BOOL)isUpdateInterfaceEnabled;

- (void)updateInterface;

- (GSGlyphsInfo *)glyphsInfo;

#pragma mark Metrics

/** The size of the grid.
 
 If it is `0`, all values are stored as floats. `1` and above will ALWAYS round the coordinates.
 
 @see gridLength
 */
@property (nonatomic, assign) NSUInteger gridMain;

/** The subdivision of the grid.
 
 Divides the grid. 
 
 @see gridLength
 */
@property (nonatomic, assign) NSUInteger gridSubDivision;

/** The effective grid width.
 
 Calculated by Grid / Subdivision
 
 @return The effective size of the grid
 */
- (CGFloat)gridLength;

@property (nonatomic, assign) CGFloat keyboardIncrement;

- (BOOL)importAFM:(NSString *)AFMString fontMaster:(nullable GSFontMaster *)FontMaster metrics:(BOOL)doMetrics kerning:(BOOL)doKerning error:(NSError *__autoreleasing *)error;

/** Recalculates the temp font.
 
 Call it, if you changed something like the glyph set or the features and need to make sure the temp font is up to date. This is usually done automatically.
 
 @param Error if there is a problem, it returns an NSError object (by reference) that contains details.
 @return YES if successfull, NO, if not.
 @see tempOTFFont
 */
- (BOOL)compileTempFontError:(out NSError *__autoreleasing *)Error;

/** The path to the temp font
 
 This font contains no outlines and no kerning but the full GSUB table. It is used for the feature preview in edit view.
 */
@property (nonatomic, strong, nullable) NSString *tempOTFFont;
@property (nonatomic, strong, nullable) COFont *compositorFont;
@property (nonatomic, strong, nullable) NSDictionary *tempProduction2NiceNames;
#endif

@property (nonatomic, strong) NSArray *displayStrings;
@property (nonatomic, strong) NSString *appVersion;
@property (nonatomic) CGFloat formatVersion;
@end
NS_ASSUME_NONNULL_END
