//
//  GlyphsReporterProtocol.h
//  GlyphsReporter
//
//  Created by Georg Seifert on 9.1.12.
//  Copyright 2012 schriftgestaltung.de. All rights reserved.
//

#import <Cocoa/Cocoa.h>

/*! GlyphsReporter: The protocol defining the reporter plugin protocol
 The pPlugin will show in the View menu and can be enabled like other view settings. It has no functionality other then draw something in the active layer.
 */

@protocol GSGlyphEditViewControllerProtocol;
@class GSLayer;

@protocol GlyphsReporter

//! The interface version.
- (NSUInteger)interfaceVersion;

//! The title of the plugin.
- (NSString *)title;

/** This method is called before anything else is done to give the plugin access to the current Edit view
 
 @param Controller The currently active edit view controller
 */
- (void)setController:(NSViewController<GSGlyphEditViewControllerProtocol> *)Controller;

@optional

/** Is called to asked the plugin to draw anything it likes behind the normal outlines
 
 @warning Deprecated in version 2.3, Use @see drawBackgroundForLayer:options:
 
 The scaling and positioning is taken care of. That means you need to draw unscaled and the scaling will be done automatically. If you need to draw something with a absulute size, you need to counter scale it.
 
 @param Layer The Layer to draw
 */
- (void)drawBackgroundForLayer:(GSLayer *)Layer;

/** Is called to asked the plugin to draw anything it likes behind the normal outlines
 
 The scaling and positioning is taken care of. That means you need to draw unscaled and the scaling will be done automatically. If you need to draw something with a absulute size, you need to counter scale it.
 
 @param Layer The Layer to draw
 @param options A dictionary with some info about the envorinment (currently "Scale")
 
 @since 2.3 (839)
 */
- (void)drawBackgroundForLayer:(GSLayer *)Layer options:(NSDictionary *)options;

/** Is called to asked the plugin to draw anything it likes behind the normal outlines
 
 implement this method if the automatic scaling gets into your way. You need to check for the current layer and position of the layer yourself.
 @since 2.3 (849)
 @param options A dictionary with some info about the envorinment (currently "Scale")
 */
- (void)drawBackgroundWithOptions:(NSDictionary *)options;

/** Implement this and return YES if you did all drawing yourself and don’t need the regular outline drawn on top of it.
 
 This is for inactive layers
 
 @param Layer the layer
 
 @return return YES if the normal outline drawing should be supressed.
 */
- (BOOL)needsExtraMainOutlineDrawingForInactiveLayer:(GSLayer *)Layer;

/** Implement this and return NO to disable the accent cloud.
 
 @param Layer the layer
 */
- (BOOL)shouldDrawAccentCloudForLayer:(GSLayer *)Layer;

/** Implement this and return YES if you did all drawing yourself and don’t need the regular outline drawn on top of it.
 
 This is for active layers
 
 @param Layer the layer
 
 @return return YES if the normal outline drawing should be supressed.
 */
- (BOOL)needsExtraMainOutlineDrawingForActiveLayer:(GSLayer *)Layer;

/** Is called to asked the plugin to draw anything it likes above the normal outlines
 
 @warning Deprecated in version 2.3, Use @see drawForegroundForLayer:options:
 
 The scaling and positioning is taken care of. That means you need to draw unscaled and the scaling will be done automatically. If you need to draw something with a absulute size, you need to counter scale it.
 
 @param Layer The Layer to draw
 */
- (void)drawForegroundForLayer:(GSLayer *)Layer;

/** Is called to asked the plugin to draw anything it likes above the normal outlines
 
 implement this method if the automatic scaling gets into your way. You need to check for the current layer and position of the layer yourself.
 @since 2.3 (849)
 
 @param options A dictionary with some info about the envorinment (currently "Scale")
 */
- (void)drawForegroundWithOptions:(NSDictionary *)options;

/** Is called to asked the plugin to draw anything it likes above the normal outlines
 
 The scaling and positioning is taken care of. That means you need to draw unscaled and the scaling will be done automatically. If you need to draw something with a absulute size, you need to counter scale it.
 
 @param Layer The Layer to draw
 @param options A dictionary with some info about the envorinment (currently "Scale")
 @since 2.3 (839)
 */
- (void)drawForegroundForLayer:(GSLayer *)Layer options:(NSDictionary *)options;

/** Is called to asked the plugin to draw anything it likes behind the normal outlines in an inactive layer

 @warning Deprecated in version 2.3, Use @see drawBackgroundForInactiveLayer:options:
 
 The scaling and positioning is taken care of. That means you need to draw unscaled and the scaling will be done automatically. If you need to draw something with a absulute size, you need to counter scale it.
 
 @param Layer The Layer to draw
 */
- (void)drawBackgroundForInactiveLayer:(GSLayer *)Layer;

/** Is called to asked the plugin to draw anything it likes behind the normal outlines in an inactive layer
 
 The scaling and positioning is taken care of. That means you need to draw unscaled and the scaling will be done automatically. If you need to draw something with a absulute size, you need to counter scale it.
 
 @param Layer The Layer to draw
 @param options A dictionary with some info about the envorinment (currently "Scale")
 @since 2.3 (839)
 */
- (void)drawBackgroundForInactiveLayer:(GSLayer *)Layer options:(NSDictionary *)options;

/** Is called to asked the plugin to draw anything it likes in front the normal outlines in an inactive layer
 
 The scaling and positioning is taken care of. That means you need to draw unscaled and the scaling will be done automatically. If you need to draw something with a absulute size, you need to counter scale it.
 
 @param Layer The Layer to draw
 @param options A dictionary with some info about the envorinment (currently "Scale")
 @since 2.5 (1120)
 */
- (void)drawForegroundForInactiveLayer:(GSLayer *)Layer options:(NSDictionary *)options;

/** Is called when a reporter is activated.
 
 Use it to run expensive operations to initialize the plugin
 */
- (void)willActivate;

/** Is called when a reporter is deactivated.
 
 Use this to clean up.
 */
- (void)willDeactivate;

/** The keyEquivalent to select the reporter.
 
 @see modifierMask
 */
- (NSString *)keyEquivalent;

/** The modifier mask that is combined with the keyEquivalent.
 
 @see keyEquivalent
 */
- (int)modifierMask;

/** add something to the context menu
 
 Will be called for active reporter plugins. This allows to have some settings.

 @param theEvent the Event that initiated the context menu. That can be used to determine mouse position.
 @param theMenu  the NSMenu where you can add your menu items.
 */
- (void)addMenuItemsForEvent:(NSEvent *)theEvent toMenu:(NSMenu *)theMenu;

- (void)addMenuItemsForEvent:(NSEvent *)theEvent controller:(NSViewController<GSGlyphEditViewControllerProtocol>*)controller toMenu:(NSMenu *)theMenu;

@end
