//
//  GlyphsToolEventProtocol.h
//  TestToolPlugin
//
//  Created by Georg Seifert on 4.7.07.
//  Copyright 2007 schriftgestaltung.de. All rights reserved.
//

#import <Cocoa/Cocoa.h>

@class GSLayer;

@protocol GSWindowControllerProtocol;
@protocol GSGlyphEditViewControllerProtocol;
@protocol GSGlyphViewControllerProtocol;

/** GlyphsToolEvent: The protocol defining the tool event plugin protocol

 All plugins should conform to this protocol. Additionaly it could conform the GlyphsToolDraw it it needs to draw.

 @see GlyphsToolDraw
 */

#define CLICK_TOLERANCE 4.0

@protocol GlyphsTool

//! The main window controller.
@property (unsafe_unretained) NSWindowController<GSWindowControllerProtocol> *windowController;
//! The current editviewcontroller controller.
@property (unsafe_unretained) NSViewController<GSGlyphEditViewControllerProtocol> *editViewController;

- (NSCursor *)standardCursor;

@end

/** Protocol for the Event handling parts of the tool API.
 
 The tool can implement all methods that are defined by [NSResponder](https://developer.apple.com/library/mac/documentation/Cocoa/Reference/ApplicationKit/Classes/NSResponder_Class/)
*/
@protocol GlyphsToolEvent

//! The interface version.
@property (nonatomic, readonly) NSUInteger interfaceVersion;

/** The group ID defines the position of the tool in the toolbar.

- selection tools (select, text): 10
- editTools (draw): 20
- navigationTools (hand, zoom): 30
*/
@property (nonatomic, readonly) NSUInteger groupID;

/// The descrition/name of the plugin.
@property (nonatomic, readonly) NSString *title;

/// The icon displayed in the toolbar.
@property (nonatomic, readonly) NSImage *toolBarIcon;

//! The default keyboard shortcut to select the tool.
@property (nonatomic, readonly) NSString *trigger;

/** The trigger (key mask) to select the tool temporarily
 
 The select tool is selected with the cmd key. It returns NSCommandKeyMask.
 */
@property (nonatomic, readonly) NSInteger tempTrigger;

/** Set to YES while the mouse is down and dragging.

The setter should initialise the drag action. (set dragStart)
*/
@property (nonatomic) BOOL dragging;

//! The position where the draggin action had started.
@property (nonatomic) NSPoint draggStart;

/** Informs the receiver that the user has pressed or released a modifier key (Shift, Control, and so on).
 
 @param anEvent An object encapsulating information about the modifier-key event.
 */
- (void)modifierChanged:(NSEvent *)anEvent;

/** Is called when the tool is activated.

This might happen when the user selects the tool or when the document becomes active
@see willDeactivate
*/
- (void)willActivate;

/** Is called when the tool is deactivated.
 
 This might happen when the user deselects the tool or when the document becomes inactive
 @see willActivate
 */
- (void)willDeactivate;

/**
 A callback to let the tool know that a temp tool will be selected (by pressing a modifier key). You can return NO to prevent that.

 @param TempTool the temp tool that is about to become active

 @return if you allow it or not
 */
- (BOOL)willSelectTempTool:(id)TempTool;

/// Draw some general things to the background of the edit view.
- (void)drawBackground;

/** Draws the active selection.
  The default is a colored dragg rect.
 */
- (void)drawSelection;

/// Draw some general things to the background of the edit view.
- (void)drawForeground;

@optional

/** Copy'n'Past Support */
- (void)copy:(id)sender;

/** Copy'n'Past Support */
- (void)paste:(id)sender;

/** Return a NSMenu object that should be displayed if the user presses/holdes the mouse on the icon

@return A NSMenu object
*/
- (NSMenu *)toolbarMenu;

- (NSResponder<GlyphsToolEvent> *)eventHandler;

- (void)mouseDoubleDown:(NSEvent *)theEvent;

- (NSMenu *)defaultContextMenu;

/**
 Callback to allow to add something to the contect menu

 @param theEvent the current event
 @param theMenu  the context menu to add to
 */
- (void)addMenuItemsForEvent:(NSEvent *)theEvent toMenu:(NSMenu *)theMenu;

- (void)addMenuItemsForEvent:(NSEvent *)theEvent controller:(NSViewController<GSGlyphEditViewControllerProtocol>*)controller toMenu:(NSMenu *)theMenu;
/**
 if the tool has several sub groups, this is called to swithch to the next tool

 @param sender the object calling the command
 */
- (void)selectNextSubTool:(id)sender;

/**
 returns all layers that arecurrently active. That is mostly used for the 'Select All Layers' tool.

 @return an array of layers
 */
- (NSArray *)activeLayers;

//! returnes the view to place in the inpector
- (NSArray *)inspectorViewControllers;

//! returnes the view to place on the left of the normal inpector
- (NSArray *)extraInspectorViewControllers;

//! return true if the current selection state might show a "Info for Selction" dialog.
- (BOOL)canShowInfoForSelection;
/** actually show the info for Selection dialog
 
 is only called if canShowInfoForSelection returned YES.
 
 @see canShowInfoForSelection
 */
- (void)showInfoForSelection;
@end

@protocol GSInspectorViewControllerDelegate <NSObject>

- (NSArray *)inspectorViewControllersForLayer:(GSLayer *)layer;

@end

@protocol GSContextMenuCallbackProtocol <NSObject>

- (void)contextMenuCallback:(NSMenu *)menu forSelectedLayers:(NSArray *)glyphs event:(NSEvent *)event;

@end
