//
//  GSComponent.h
//  Glyphs
//
//  Created by Georg Seifert on 30.3.08.
//  Copyright 2008 schriftgestaltung.de. All rights reserved.
//

#import <Cocoa/Cocoa.h>
#import "GSElement.h"
@class GSGlyph;
@class GSFont;
@class GSLayer;

/*! \brief GSComponent: The class defining the component object
 \ingroup common
 This class defines the component object. It is a subclass of GSElement
 */
@interface GSComponent : GSElement {
	NSString * componentName;
	GSGlyph * component;
	@public NSAffineTransformStruct transformStruct;
	NSString * anchor;
	BOOL disableAlignment;
	BOOL locked;
}
/** components are referenced by the glyph name.

 If you set the componentName the component value is set and this is used during runtime. So you can change the glyph name of the base glyph during runtime.
 */
@property(strong, nonatomic) NSString * componentName;

/** The base glyph

 This is readonly. To set the base glyph use componentName: newBaseGlpyh.name.
 @see componentName
 */
@property(unsafe_unretained, readonly) GSGlyph * component;

/** The transformation of the component.

 The values of the struct are:
 {m11, m12, m21, m22, tX, tY}
 this is equivalent to the glif attributes
 {xScale, xyScale, yxScale, yScale, xOffset, yOffset}
 in the same order.
 The initial value is {1,0,0,1,0,0}.
 */
@property(assign, nonatomic) NSAffineTransformStruct transformStruct;

@property(copy, nonatomic) NSString * anchor;

@property (nonatomic, assign) BOOL disableAlignment;
@property (nonatomic) BOOL locked;
- (NSRect) bounds;

/// The bezierPath of the component. It is already transformed.
- (NSBezierPath*) bezierPath;
/// The bezierPath of open paths of the component. It is already transformed.
- (NSBezierPath*) openBezierPath;
- (NSData*) bezierData;

- (void) updateComponentGlyph;
#ifndef GLYPHS_VIEWER
- (void) setScaleX:(CGFloat) sX scaleY:(CGFloat) sY rotation:(CGFloat) R;

/** Decomposes the component to the containing Layer

 It need to parent property set correctly. It calls [parent decomposeComponent:self].
 */
- (void) decompose;
#ifndef GLYPHS_LITE
- (BOOL) intersects:(GSComponent*)OtherComponent ;
#endif
/** It makes the component the first in the containing Layer

 It need to parent property set correctly. It calls [parent makeFirstComponent:self].
 */
- (void) makeFirst;
- (void) makeDisableAlignment;
- (void) makeEnableAlignment;
#endif
@end
