//
//  GSGlyph.h
//  FontBezier Doc
//
//  Created by Georg Seifert on 20.10.05.
//  Copyright 2005 schriftgestaltung.de. All rights reserved.
//

#import <Cocoa/Cocoa.h>

//#import "GlyphIconCollectionViewItem.h"
//#import "GlyphIconView.h"


@class GSFont;
@class GSLayer;
@class GSUndoManager;
@class MGOrderedDictionary;

/*! \brief GSGlyph: The class defining the glyph object
 \ingroup common
 This class defines the glyph object.
 */

@interface GSGlyph : NSObject <NSCoding, NSCopying> {
	MGOrderedDictionary		*_layers;
	NSString				*_name;
	NSString				*_unicode;
	UTF32Char				_unicodeChar;
	NSString				*_script;
	NSString				*_note;
	GSFont					*__unsafe_unretained _parent;
	//NSImage* image;
	NSArray					*_baseGlyphs;
	NSString				*_category;
	NSString				*_subCategory;
	NSString				*_leftMetricsKey;
	NSString				*_widthMetricsKey;
	NSString				*_rightMetricsKey;
	NSString				*_vertWidthMetricsKey;
	NSString				*_topMetricsKey;
	NSString				*_bottomMetricsKey;

	NSString				*_leftKerningGroup;
	NSString				*_rightKerningGroup;
	NSString				*_leftKerningGroupId;
	NSString				*_rightKerningGroupId;
	
	NSString				*_topKerningGroup;
	NSString				*_bottomKerningGroup;
	NSString				*_topKerningGroupId;
	NSString				*_bottomKerningGroupId;
	
	NSUInteger				_glyphId;
	NSString				*_id;
	BOOL					_export;
	NSColor					*_color;
	NSInteger				_colorIndex;
	NSMutableDictionary		*_userData;
	NSDate					*_lastChange;
	//NSMutableDictionary		*changeCount;
	// caching the objects for the Collection View.
//#ifndef GLYPHS_LITE
//	id _item;
//#endif
	//id _itemView;
	NSInteger				_changeCount;
#ifndef GLYPHS_VIEWER
	GSUndoManager *			_undoManager;
#endif
//	NSDictionary *			_tempGlyphDict;
	NSString *				_sortName;
	NSString *				_sortNameKeep;
	NSString *				_legacy;
	NSDictionary *			glyphDict;
}
/// @name Properties
/// A pointer to the containing GSFont object
@property(unsafe_unretained, nonatomic) GSFont * parent;
@property(readonly, nonatomic) GSFont * font;
@property(assign, nonatomic) NSInteger changeCount;

/// An array of GSLayer objects. (it should have at least the same count than font.fontMasters)
@property(readonly, nonatomic) NSDictionary * layers;

/// The glyphs name
@property(nonatomic, copy) NSString * name;
/// The unicode value of the glyph. This is normaly set automaticaly on setting the name
@property(nonatomic, copy) NSString * unicode;
@property(nonatomic, readonly) UTF32Char unicodeChar;
/// The glyphs note
@property(nonatomic, copy) NSString * note;
/** The language/script of the glyph.

 e.g: latin, cyrillic, greek
 This is set from the internal database.
 */
@property(nonatomic, copy) NSString * script;
/** The category of the glyph

 e.g: letter, number, punctuation ...
 this is set from the internal database.
 */
@property(nonatomic, copy) NSString * category;

/** The sub category of the glyph

 e.g: lower case, smallcaps ...
 this is set from the internal database.
 */
@property(nonatomic, copy) NSString * subCategory;
/// The glyphs the glyph is composed of.
@property(nonatomic, strong) NSArray * baseGlyphs;
#ifndef GLYPHS_VIEWER
/** The content of the glyph as dictionary to store it in a pList.

 This is used to store the data in the .glyphs file.
 */
@property(unsafe_unretained, readonly) NSDictionary* glyphDict;
/// The glyphname of the keyglyph for LSB.
@property(copy, nonatomic) NSString* leftMetricsKey;
@property(copy, nonatomic) NSString* widthMetricsKey;
@property(copy, nonatomic) NSString* vertWidthMetricsKey;
/// The glyphname of the keyglyph for RSB.
@property(copy, nonatomic) NSString* rightMetricsKey;
@property(copy, nonatomic) NSString* topMetricsKey;
@property(copy, nonatomic) NSString* bottomMetricsKey;
#endif
/// The name of the kerning group for left side kerning.
@property(copy, nonatomic) NSString* leftKerningGroup;
/// The name of the kerning group for right side kerning.
@property(copy, nonatomic) NSString* rightKerningGroup;

/** The id of the keyglyph for left side kerning.

 this sould look like this: `@MM_L_Name`
 */
@property(copy, nonatomic) NSString* leftKerningGroupId;
/** The id of the keyglyph for right side kerning.

 this sould look like this: `@MM_R_Name`
 */
@property(copy, nonatomic) NSString* rightKerningGroupId;

@property(copy, nonatomic) NSString* topKerningGroup;
@property(copy, nonatomic) NSString* bottomKerningGroup;
@property(copy, nonatomic) NSString* topKerningGroupId;
@property(copy, nonatomic) NSString* bottomKerningGroupId;

#ifndef __cplusplus //RMX complains about that.
/// If the glyph will end up in the final font
@property(nonatomic)	BOOL export;
#endif
/// The NSColor object of the color label
@property(unsafe_unretained, readonly, nonatomic) NSColor *color;
/** Returns the index of the color label

 @return 0–11, anything else means no label
 */
@property(assign, nonatomic) NSInteger colorIndex;
/** Place to store custom data.

 Here it is possible to store something. Please use a unique key.
 The objects should be able to serialize to a plist. If they are not, they should return all infos to store in description (as String).
 */
@property(nonatomic, strong) NSMutableDictionary * userData;

//@property		BOOL export;
/** The glyphs NSUndoManager 

 Each GSGlyph object has its own undoManager
 @return a NSUndoManager object
 */
#ifndef GLYPHS_VIEWER
@property(strong, nonatomic) NSUndoManager *undoManager;
#endif
@property(nonatomic) NSUInteger glyphId;

/** The glyph key is used to access the glyph.

 This is used to keep track of the glyph even if is renamed.
 It is used for the kerning.
 It could look like this: "FBCA074D-FCF3-427E-A700-7E318A949AE5"
 */
@property(strong, nonatomic) NSString * id;
@property(copy, nonatomic) NSString *sortName;
@property(copy, nonatomic) NSString *sortNameKeep;
@property(copy, nonatomic) NSString *legacy;
@property(nonatomic, strong) NSDate *lastChange;
- (NSString*) charString;
//@property(retain) id itemView;
/// Initialisation
/// Initializes a newly allocated glyph with this name
//- (id) initWithName:(NSString*) Name ;

/** initialises a glyph with a dictionary loaded from a pList.

 @param glyphDict A dictionary
 */
- (id) initWithGlyphDict:(NSDictionary*) glyphDict ;
#ifndef GLYPHS_VIEWER
- (BOOL) saveToFile:(FILE*) File error:(NSError**) error ;
#endif
/// @name Methods
/** Sets the Changecount of the Glyph.
 
 to manualy clean up the undo count.
 */
#ifndef GLYPHS_VIEWER
- (void) updateChangeCount:(NSDocumentChangeType)changeType ;

- (NSError*) validateName:(NSString*)Name ;
#endif
- (void) setName:(NSString*) Name changeName:(BOOL) ChangeName ;
- (void) setName:(NSString *)Name changeName:(BOOL) ChangeName update:(BOOL) Update ;
#pragma Layers
- (NSUInteger) countOfLayers;
/** Returns the a layer loacated at a Index.

 @param Key The Id of the layer
 @return A GSLayer object.
*/
- (GSLayer *) layerForKey:(NSString*) Key ;
- (GSLayer *) layerForName:(NSString*) Key;
/** Replaces the layer at index with Layer

 @param Layer The GSLayer object.
 @param Index An index within the bounds of the the layers array.
 */
- (NSString*) keyForLayer:(GSLayer*) Layer ;
- (void) setLayers:(NSDictionary *)value;
/** Sets the with the key.

 @param Layer The GSLayer object.
 @param Key The layer key. For master layers this is the id of the corresponding GSFontMaster.
 */
- (void) setLayer:(GSLayer*) Layer forKey:(id) Key;
#ifndef GLYPHS_VIEWER
/** Removes the Layer for the key.

 @param Key The layer key.
 */
- (void) removeLayerForKey:(id) Key ;
/** Returns if the \a layers have compatibel outlines.

 @return YES if the layers have compatible outlines.
 */

- (BOOL) mastersCompatible ;
/** Generates a interplated Layer

 It interpolates all layers according to the settings in Interpolation
 @param Interpolation
 */
- (GSLayer*) interpolate:(NSDictionary*) Interpolation NS_DEPRECATED_MAC(10_0, 10_0);
- (GSLayer*) interpolate:(NSDictionary*) Interpolation error:(NSError**) error NS_DEPRECATED_MAC(10_0, 10_0);
- (GSLayer*) interpolate:(NSDictionary*) Interpolation decompose:(BOOL)Decompose error:(NSError**) error ;
/// Is called from other object after changing some of the glyphs properties.
- (void) outlineHasChanges ;
- (id) copyThin:(BOOL) Thin layers:(BOOL) Layers ;
#endif
@end
