//
//  GSFontMaster.h
//  Glyphs
//
//  Created by Georg Seifert on 6.9.07.
//  Copyright 2007 schriftgestaltung.de. All rights reserved.
//

#import <Cocoa/Cocoa.h>

@class GSFont;
@class GSGuideLine;
@class GSAlignmentZone;

/** The class defining the fontmaster object */
@interface GSFontMaster : NSObject <NSCoding, NSCopying> {
	//NSInteger _index;
	GSFont				*_font;
	CGFloat				_xHeight;
	CGFloat				_descender;
	CGFloat				_ascender;
	CGFloat				_capHeight;
	CGFloat				_italicAngle;
	NSMutableArray		*_verticalStems, *_horizontalStems;
	NSMutableArray		*_alignmentZones;
	NSMutableDictionary	*_userData;
	NSString			*_weight;
	NSString			*_width;
	NSString			*_custom;
	NSString			*_id;
	CGFloat				_weightValue, _widthValue, _customValue;
	NSMutableArray		*_guideLines;
	NSMutableDictionary	*_changeCount;
	NSMutableIndexSet	*_alignmentZoneCache;
	NSMutableArray *	_customParameters;
}
@property(assign, nonatomic) GSFont* font;
/** @name Properties */

/** the name of the FontMaster
 e.g; "Regular", "Condensed", ...
 */
@property(readonly, nonatomic) NSString* name;

/** The width tag

 Something like Condensed or Extended
 */
@property(copy, nonatomic) NSString* width;

/** The weight tag
 
 Something like Light or Regular
 */
@property(copy, nonatomic) NSString* weight;

/** A custom tag */
@property(copy) NSString* custom;

/** The weight value

 The weight value defines the place of the Fontmaster in the "design space"
 */
@property(assign, nonatomic) CGFloat weightValue;

/** The width value

 The width value defines the place of the Fontmaster in the "design space"
 */
@property(assign, nonatomic) CGFloat widthValue;

/** The custom value
 
 The custom value defines the place of the Fontmaster in the "design space"
 */
@property(assign) CGFloat customValue;

/// The id
@property(copy, nonatomic) NSString* id;

/// The descender.
@property (nonatomic) CGFloat descender;

/// The ascender.
@property (nonatomic) CGFloat ascender;

/// The capHeight.
@property (nonatomic) CGFloat capHeight;
/// The xHeight.
@property (nonatomic) CGFloat xHeight;
/// The Italic angle
@property (nonatomic) CGFloat italicAngle;
/// list of vertical stems
@property (retain, nonatomic) NSMutableArray * verticalStems;
/// list of horizontal stems
@property (retain, nonatomic) NSMutableArray * horizontalStems;
/// list of alignment zones
@property (retain, nonatomic) NSArray * alignmentZones;
/// list of guidelines
@property (retain, nonatomic) NSArray * guideLines;
/** Place for plugins to store data .

 Here it is possible to store something. Please use a unique key.
 The objects should be able to serialize to a plist. if they are not they should return all infos to store to save as string in description.
 */
@property(retain, nonatomic) NSMutableDictionary * userData;
@property(retain, nonatomic) NSMutableArray *customParameters;
/** @name Initializing an Font Master */

/// Inits with a Dict loaded from pList.
- (id) initWithFontMasterDict:(NSDictionary *) FontMasterDict;
/// Returnes the Font Master to save it in plist
- (NSDictionary*) fontMasterDict;
- (BOOL) saveToFile:(FILE*) File error:(NSError**) error ;
/** @name Guidelines */

/** Number of Guide Lines
 
 @return Number of Guide Lines 
 */
- (NSUInteger) guideLineCount ;

/** Returns the Guide Line located at Index.
 @param Index An index within the bounds of the array.
 @return The Guide Line located at index.
 */
- (id) guideLineAtIndex:(NSUInteger)theIndex ;
- (void) getGuideLines:(id *)objsPtr range:(NSRange)range ;
- (void) addGuideLine:(GSGuideLine*)GuideLine ;
- (void) removeGuideLine:(GSGuideLine*) GuideLine ;
- (void) insertGuideLine:(GSGuideLine*)GuideLine atIndex:(NSUInteger)theIndex ;
- (void) removeGuideLineAtIndex:(NSUInteger)theIndex ;

/** @name Alignment Zones */
/** Number of alignment zones
 
 @return Number of alignment zones 
 */
- (NSUInteger) countOfAlignmentZones ;

/** Returns the alignment zone located at Index.
 @param Index An index within the bounds of the array.
 @return The object located at index.
 */
- (GSAlignmentZone*) alignmentZoneAtIndex:(NSUInteger)Index ;
/** adds an alignment zone
 @param AlignmentZone a GSAlignmentZone object
 */
- (void) addAlignmentZone: (GSAlignmentZone*) AlignmentZone ;
- (void) addAlignmentZones: (NSArray*) AlignmentZones ;
- (void) removeAlignmentZone: (GSAlignmentZone*) AlignmentZone ;
- (void) removeAlignmentZones: (NSArray*) AlignmentZones ;
- (void) removeAlignmentZoneAtIndex:(NSUInteger)theIndex ;
- (void) removeAlignmentZoneAtIndexes:(NSIndexSet*)theIndexes ;
- (void) sortAlignmentZones;
- (NSArray*) blueValues;
- (NSArray*) otherBlues;
- (BOOL) isInAlignmentZone: (NSInteger) Height ;
- (BOOL) isOnVerticalMetrics: (NSInteger) Height ;
- (NSUInteger)countOfCustomParameters;
- (id) objectInCustomParametersAtIndex:(NSUInteger)theIndex;
- (void) getCustomParameters:(id *)objsPtr range:(NSRange)range;
- (void) insertObject:(id)obj inCustomParametersAtIndex:(NSUInteger)theIndex;
- (void) removeObjectFromCustomParametersAtIndex:(NSUInteger)theIndex;
//- (void) addValueToCustomParameters:(id) Value forKey:(NSString*) Key;
- (void) removeObjectFromCustomParametersForKey:(NSString*) Key;
/** @name Other Methods */
/** Scales the Font Master
 
 Scales the vertical metrics, alignment zones and guide lines
 @param Scale the scale factor (1 means no scaling.)
 */
- (void) scaleBy:(CGFloat) Scale ;

@end
